(*
 * SNU 4190.310 Programming Languages (Fall 2006)
 *
 * K- Interpreter I
 *)

open K

structure KParseTreePrinter : sig val print : program -> unit end =
  struct
    fun quote x = ["\"" ^ x ^ "\""]

    fun indent l = List.map (fn s => "  " ^ s) l

    fun comma [] = []
      | comma [h] = [h ^ ", "]
      | comma (h :: t) = h :: (comma t)

    fun tree cons [] = [cons]
      | tree cons (h :: t) =
        (cons ^ " (")
          :: (List.fold_left (fn l x => (comma l) @ (indent x)) (indent h) t)
          @ [(")")]

    fun tree_strs e =
        case e of
          NUM n => tree "NUM" [[string_of_int n]]
        | TRUE => tree "TRUE" []
        | FALSE => tree "FALSE" []
        | UNIT => tree "UNIT" []
        | VAR x => tree "VAR" [quote x]
        | ADD (e1, e2) => tree "ADD" [tree_strs e1, tree_strs e2]
        | SUB (e1, e2) => tree "SUB" [tree_strs e1, tree_strs e2]
        | MUL (e1, e2) => tree "MUL" [tree_strs e1, tree_strs e2]
        | DIV (e1, e2) => tree "DIV" [tree_strs e1, tree_strs e2]
        | EQUAL (e1, e2) => tree "EQUAL" [tree_strs e1, tree_strs e2]
        | LESS (e1, e2) => tree "LESS" [tree_strs e1, tree_strs e2]
        | NOT e => tree "NOT" [tree_strs e]
        | ASSIGNV (x, e) => tree "ASSIGNV" [quote x, tree_strs e]
        | ASSIGNF (e1, x, e2) =>
          tree "ASSIGNF" [tree_strs e1, quote x, tree_strs e2]
        | ASSIGNG (e1, e2) => tree "ASSIGNG" [tree_strs e1, tree_strs e2]
        | SEQ (e1, e2) => tree "SEQ" [tree_strs e1, tree_strs e2]
        | IF2 (e1, e2, e3) =>
          tree "IF2" [tree_strs e1, tree_strs e2, tree_strs e3]
        | IF1 (e1, e2) => tree "IF1" [tree_strs e1, tree_strs e2]
        | WHILE (e1, e2) => tree "WHILE" [tree_strs e1, tree_strs e2]
        | FOR (x, e1, e2, e3) =>
          tree "FOR" [quote x, tree_strs e1, tree_strs e2, tree_strs e3]
        | LETV (x, e1, e2) =>
          tree "LETV" [quote x, tree_strs e1, tree_strs e2]
        | LETF (f, x, e1, e2) =>
          tree "LETF" [quote f, quote x, tree_strs e1, tree_strs e2]
        | CALLV (f, e) => tree "CALLV" [quote f, tree_strs e]
        | CALLR (f, y) => tree "CALLR" [quote f, quote y]
        | RECORD fields =>
          tree
            "RECORD"
            (List.map (fn (x, e) => tree ""[quote x, tree_strs e]) fields)
        | FIELD (e, x) => tree "FIELD" [tree_strs e, quote x]
        | MALLOC e => tree "MALLOC" [tree_strs e]
        | AMPER x => tree "AMPER" [quote x]
        | STAR e => tree "STAR" [tree_strs e]
        | READ x => tree "READ" [quote x]
        | WRITE e => tree "WRITE" [tree_strs e]

    fun print pgm =  List.iter print_endline (tree_strs pgm)
  end
